function Proteomap(f,names,textscale)
%Turn off a warning that occurs during this script (warning does not affect
%outcome)
warning('off','MATLAB:polyshape:repairedBySimplify')


%Validate input proteome vector structure: 
%if a matrix is supplied, return an error
%if a row vector is supplied, convert it to a column vector
if size(f,1) == 1 && size(f,2) > 1
   f = f';
elseif size(f,1) > 1 && size(f,2) > 1
   error('Invalid input: 2d matrix')
end

%Get number of sectors
n = length(f);

%Initialize the "seed points" for the Voronoi diagram in a circle
%(proteomap is first equally divided as a "pie")
p_0 = [0.5+0.5*cos(0.5*pi-2*pi*(1:n)/n)'  0.5+0.5*sin(0.5*pi-2*pi*(1:n)/n)'];
p_0 = p_0 + 0.01*rand(n,2) - 0.005; %randomize a little bit, else the solver gets stuck

%Find the set of seed points such that the area of each polygon in the voronoi diagram matches the
%proteome sector size
options = optimset('TolFun',1e-3);
[p_solved,res] = fsolve(@(p) areadif(p,f),p_0,options);

%If the "textscale" input is not provided, set it to 1 by default
if ~exist('textscale','var')
    textscale = 1;
end

%Draw the proteomap itself, and turn on te grid
VoronoiDraw(p_solved,names,textscale,f)
grid

%Turn on the warning that was turned off at the start
warning('on','MATLAB:polyshape:repairedBySimplify')
end

%Function that returns the difference between the areas of the polygons and the proteome sector sizes 
function dA = areadif(p,A_r)

%Validate input: proteome must be normalized 
if size(p,1) ~= size(A_r)
    error('Number of points does not match number of areas')
elseif round(sum(A_r),4) ~= 1
    error('Sum of areas is not equal to 1')
end

%Get the set of polygons (as polyshape objects) that make the voronoi diagram of the points
polys = VoronoiPolygons(p);

%Calculate the areas of all polygons
for i = 1:numel(polys)
    A(i,1) = area(polys{i});
end

%Return the differences between area sizes and proteome sector sizes (solver tries to make it 0) 
dA = A - A_r;
end

%Function that draws the diagram itself
function VoronoiDraw(p,names,textscale,f)

    %Get the set of polygons (as polyshape objects) that make the voronoi diagram of the point
    voronoipolys = VoronoiPolygons(p);
    
    %Set colormap
    cmap = parula(numel(voronoipolys));
    
    %Draw polygons
    for i = 1:numel(voronoipolys)
    
    plot(voronoipolys{i},'FaceColor',cmap(i,:))
    xlim([0 1]); xticks([0 0.1 0.2 0.3 0.4 0.5 0.6 0.7 0.8 0.9 1])
    ylim([0 1]); yticks([0 0.1 0.2 0.3 0.4 0.5 0.6 0.7 0.8 0.9 1])
    axis square
    hold on 
    
    %Plot proteome sector name at centroid of polygons, 
    %fontsize depends on ares    
    [xbar,ybar] = centroid(voronoipolys{i});
    A = area(voronoipolys{i});
    text(xbar,ybar,names{i},'FontSize',max(400*sqrt(A)*textscale,1),'HorizontalAlignment','center','VerticalAlignment','bottom')
    text(xbar,ybar,[num2str(round(100*f(i),2)) '%'],'FontSize',max(200*sqrt(A)*textscale,1),'HorizontalAlignment','center','VerticalAlignment','top')
    end
    
end

%Function that returns the polygons (as polyshape objects) of the voronoi diagram defined by the
%set of points in p
function voronoipolys = VoronoiPolygons(p)
%Separate p in x and y
x = p(:,1); y = p(:,2);

%Make delaunay triangulation of point set ("connect neighboring points")
DT = delaunayTriangulation(p);

%Calculate the vertices of the voronoi cells from this triangulation
[v,c] = voronoiDiagram(DT);

%Cut off infinite edges such that only a 1x1 square counts
F = freeBoundary(DT);       %Get outermost point set of seed point cloud (= cells that border edge of figure)
for i = 1:size(F,1)
    %Get midpoint perpendicular between outermost points (= the voronoi cell edge that intersect figure boundary) 
    a = [x(F(i,1)); y(F(i,1))];
    b = [x(F(i,2)); y(F(i,2))];
    s = 0.5*(a+b);
    r = [0 -1; 1 0] * (b-a);
    
    %Get intersections of midpoint perpendicular with boundaries, and
    %calculate the closest voronoi cell
    p_b = [ (s + ((1-s(2))/r(2))*r)';  %North boundary
            (s + ((1-s(1))/r(1))*r)';  %East boundary
            (s + (-s(2)/r(2))*r)';     %South boundary
            (s + (-s(1)/r(1))*r)'];    %West boundary
    nnv = nearestNeighbor(DT,p_b);
    
    %Only add the points that are not closer to a third point
    for j = 1:length(p_b)
        if ismember(nnv(j),F(i,:))
            v(end+1,:) = p_b(j,:);
            c{F(i,1)}(end+1) = size(v,1);
            c{F(i,2)}(end+1) = size(v,1);
        end
    end
    
end

%Add corners of figure to appropriate cells
x_b = [0; 1; 1; 0; 0]; y_b = [0; 0; 1; 1; 0];
nnv = nearestNeighbor(DT,[x_b y_b]);
for i = 1:4
    v(end+1,:) = [x_b(i) y_b(i)];
    c{nnv(i)}(end+1) = size(v,1);
end

%Make polyshape object for figure edge
bnds = polyshape(x_b,y_b);

for i = 1:size(p,1)
    %Remove indices of vertices at infinity from list
    c{i}(find(c{i}==1)) = [];
    
    %Make polyshape object of convex hull of all vertices of a voronoi
    %cell, and make sure it is bounded by the figure
    voronoipolys{i} = intersect(convhull(polyshape(v(c{i},1),v(c{i},2))),bnds);
end
end